'use client';

import React, { useState, useEffect } from 'react';
import { motion } from 'framer-motion';

interface AnimatedTextProps {
    text: string;
}

const targetDate = new Date("2025-04-25T00:00:00"); // April 25, 2025 at 12:00:00

export default function AnimatedText({ text }: AnimatedTextProps) {
    const calculateTimeLeft = () => {
        const now = new Date();
        const difference = targetDate.getTime() - now.getTime();
        let timeLeft = {
            days: 0,
            hours: 0,
            minutes: 0,
            seconds: 0,
        };

        if (difference > 0) {
            timeLeft = {
                days: Math.floor(difference / (1000 * 60 * 60 * 24)),
                hours: Math.floor((difference / (1000 * 60 * 60)) % 24),
                minutes: Math.floor((difference / (1000 * 60)) % 60),
                seconds: Math.floor((difference / 1000) % 60),
            };
        }

        return timeLeft;
    };

    const [timeLeft, setTimeLeft] = useState(calculateTimeLeft());

    useEffect(() => {
        let mounted = true;

        const timer = setInterval(() => {
            if (mounted) {
                setTimeLeft(calculateTimeLeft());
            }
        }, 1000);

        return () => {
            mounted = false;
            clearInterval(timer);
        };
    }, []);

    return (
        <div className="absolute inset-0 pb-[100px] md:pb-[120px] flex items-end justify-center">
            <div className="relative z-10 text-center text-white">
                <motion.h1
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ duration: 1 }} className="font-script text-4xl md:text-6xl">
                    {text}
                </motion.h1>

                <motion.div
                    initial={{ opacity: 0 }}
                    animate={{ opacity: 1 }}
                    transition={{ delay: 0.5, duration: 1 }}
                    className="space-y-2 md:space-y-4 mt-2 md:mt-4"
                >
                    <div className="flex justify-center items-center space-x-4 md:space-x-8 font-light">
                        <div>
                            <p className="text-3xl md:text-3xl font-script">{timeLeft.days}</p>
                            <p className='text-sm md:text-base'>Days</p>
                        </div>
                        <div className="border-l border-white h-6"></div>
                        <div>
                            <p className="text-3xl md:text-3xl font-script">{timeLeft.hours}</p>
                            <p className='text-sm md:text-base'>Hours</p>
                        </div>
                        <div className="border-l border-white h-6"></div>
                        <div>
                            <p className="text-3xl md:text-3xl font-script">{timeLeft.minutes}</p>
                            <p className='text-sm md:text-base'>Minutes</p>
                        </div>
                        <div className="border-l border-white h-6"></div>
                        <div>
                            <p className="text-3xl md:text-3xl font-script">{timeLeft.seconds}</p>
                            <p className='text-sm md:text-base'>Seconds</p>
                        </div>
                    </div>
                </motion.div>
            </div>
        </div>
    );
} 